const CLIPDROP_API_KEY = import.meta.env.VITE_CLIPDROP_API_KEY;
const CLIPDROP_API_URL = 'https://clipdrop-api.co/remove-background/v1';

export interface RemoveBackgroundResponse {
  success: boolean;
  imageUrl?: string;
  error?: string;
}

export async function removeBackground(file: File): Promise<RemoveBackgroundResponse> {
  if (!CLIPDROP_API_KEY) {
    return {
      success: false,
      error: 'ClipDrop API key is not configured. Please add VITE_CLIPDROP_API_KEY to your .env file.'
    };
  }

  try {
    const formData = new FormData();
    formData.append('image_file', file);

    const response = await fetch(CLIPDROP_API_URL, {
      method: 'POST',
      headers: {
        'x-api-key': CLIPDROP_API_KEY,
      },
      body: formData,
    });

    if (!response.ok) {
      const errorText = await response.text();
      return {
        success: false,
        error: `API Error: ${response.status} - ${errorText || 'Failed to process image'}`
      };
    }

    const blob = await response.blob();
    const imageUrl = URL.createObjectURL(blob);

    return {
      success: true,
      imageUrl
    };
  } catch (error) {
    return {
      success: false,
      error: error instanceof Error ? error.message : 'Unknown error occurred'
    };
  }
}

export function validateImageFile(file: File): { valid: boolean; error?: string } {
  const validTypes = ['image/jpeg', 'image/png', 'image/webp'];
  const maxSize = 10 * 1024 * 1024;

  if (!validTypes.includes(file.type)) {
    return {
      valid: false,
      error: 'Invalid file type. Please upload a JPG, PNG, or WEBP image.'
    };
  }

  if (file.size > maxSize) {
    return {
      valid: false,
      error: 'File size exceeds 10MB limit. Please upload a smaller image.'
    };
  }

  return { valid: true };
}
