import { Trash2, Download } from 'lucide-react';

export interface ProcessedImage {
  id: string;
  originalUrl: string;
  processedUrl: string;
  timestamp: Date;
  filename: string;
}

interface ImageGalleryProps {
  images: ProcessedImage[];
  onDelete: (id: string) => void;
  onDownload: (image: ProcessedImage) => void;
}

export default function ImageGallery({ images, onDelete, onDownload }: ImageGalleryProps) {
  if (images.length === 0) {
    return null;
  }

  return (
    <div className="space-y-4">
      <div className="flex items-center justify-between">
        <h2 className="text-2xl font-bold text-gray-800">Recent Processed Images</h2>
        <span className="text-sm text-gray-500">{images.length} image{images.length !== 1 ? 's' : ''}</span>
      </div>

      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4">
        {images.map((image) => (
          <div
            key={image.id}
            className="group relative bg-white rounded-lg shadow-md hover:shadow-xl transition-shadow overflow-hidden border border-gray-200"
          >
            <div className="aspect-square bg-gradient-to-br from-gray-100 to-gray-200 relative overflow-hidden">
              <div className="absolute inset-0 bg-[linear-gradient(45deg,#f0f0f0_25%,transparent_25%,transparent_75%,#f0f0f0_75%,#f0f0f0),linear-gradient(45deg,#f0f0f0_25%,transparent_25%,transparent_75%,#f0f0f0_75%,#f0f0f0)] bg-[length:20px_20px] bg-[position:0_0,10px_10px]" />
              <img
                src={image.processedUrl}
                alt={image.filename}
                className="relative w-full h-full object-contain p-2"
              />
            </div>

            <div className="p-3 space-y-2">
              <p className="text-sm font-medium text-gray-700 truncate" title={image.filename}>
                {image.filename}
              </p>
              <p className="text-xs text-gray-500">
                {image.timestamp.toLocaleString()}
              </p>

              <div className="flex space-x-2 pt-2">
                <button
                  onClick={() => onDownload(image)}
                  className="flex-1 flex items-center justify-center space-x-1 px-3 py-2 bg-blue-600 hover:bg-blue-700 text-white text-sm font-medium rounded transition-colors"
                  title="Download"
                >
                  <Download className="w-4 h-4" />
                  <span>Download</span>
                </button>
                <button
                  onClick={() => onDelete(image.id)}
                  className="px-3 py-2 bg-red-600 hover:bg-red-700 text-white rounded transition-colors"
                  title="Delete"
                >
                  <Trash2 className="w-4 h-4" />
                </button>
              </div>
            </div>
          </div>
        ))}
      </div>
    </div>
  );
}
