import { useState } from 'react';
import { AlertCircle, Scissors } from 'lucide-react';
import FileUpload from './components/FileUpload';
import ImageComparison from './components/ImageComparison';
import ImageGallery, { ProcessedImage } from './components/ImageGallery';
import { removeBackground, validateImageFile } from './utils/clipdrop';
import { downloadImage, resizeImage, generateFilename } from './utils/imageHelpers';
import { useLocalStorage } from './hooks/useLocalStorage';

function App() {
  const [originalImage, setOriginalImage] = useState<string | null>(null);
  const [processedImage, setProcessedImage] = useState<string | null>(null);
  const [isProcessing, setIsProcessing] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [currentFilename, setCurrentFilename] = useState<string>('');
  const [gallery, setGallery] = useLocalStorage<ProcessedImage[]>('bg-remover-gallery', []);

  const handleFileSelect = async (file: File) => {
    setError(null);

    const validation = validateImageFile(file);
    if (!validation.valid) {
      setError(validation.error || 'Invalid file');
      return;
    }

    setCurrentFilename(file.name);
    const originalUrl = URL.createObjectURL(file);
    setOriginalImage(originalUrl);
    setProcessedImage(null);
    setIsProcessing(true);

    try {
      let fileToProcess = file;

      if (file.size > 5 * 1024 * 1024) {
        try {
          fileToProcess = await resizeImage(file);
        } catch (resizeError) {
          console.warn('Image resize failed, using original:', resizeError);
        }
      }

      const result = await removeBackground(fileToProcess);

      if (result.success && result.imageUrl) {
        setProcessedImage(result.imageUrl);

        const newImage: ProcessedImage = {
          id: Date.now().toString(),
          originalUrl,
          processedUrl: result.imageUrl,
          timestamp: new Date(),
          filename: file.name
        };

        setGallery(prev => [newImage, ...prev].slice(0, 20));
      } else {
        setError(result.error || 'Failed to process image');
      }
    } catch (err) {
      setError(err instanceof Error ? err.message : 'An unexpected error occurred');
    } finally {
      setIsProcessing(false);
    }
  };

  const handleDownload = () => {
    if (processedImage) {
      const filename = generateFilename(currentFilename);
      downloadImage(processedImage, filename);
    }
  };

  const handleGalleryDownload = (image: ProcessedImage) => {
    const filename = generateFilename(image.filename);
    downloadImage(image.processedUrl, filename);
  };

  const handleGalleryDelete = (id: string) => {
    setGallery(prev => prev.filter(img => img.id !== id));
  };

  const handleReset = () => {
    setOriginalImage(null);
    setProcessedImage(null);
    setError(null);
    setCurrentFilename('');
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 to-slate-100">
      <div className="container mx-auto px-4 py-8 max-w-7xl">
        <header className="text-center mb-12">
          <div className="flex items-center justify-center space-x-3 mb-4">
            <Scissors className="w-10 h-10 text-blue-600" />
            <h1 className="text-4xl font-bold text-gray-800">Background Remover</h1>
          </div>
          <p className="text-lg text-gray-600 max-w-2xl mx-auto">
            Remove backgrounds from your images instantly using AI technology.
            Simply upload an image and watch the magic happen.
          </p>
        </header>

        {error && (
          <div className="mb-6 p-4 bg-red-50 border border-red-200 rounded-lg flex items-start space-x-3">
            <AlertCircle className="w-5 h-5 text-red-600 flex-shrink-0 mt-0.5" />
            <div>
              <p className="text-red-800 font-semibold">Error</p>
              <p className="text-red-700 text-sm">{error}</p>
            </div>
          </div>
        )}

        <div className="bg-white rounded-xl shadow-lg p-8 mb-12">
          {!originalImage ? (
            <FileUpload onFileSelect={handleFileSelect} disabled={isProcessing} />
          ) : (
            <div className="space-y-6">
              <ImageComparison
                originalImage={originalImage}
                processedImage={processedImage}
                isProcessing={isProcessing}
                onDownload={handleDownload}
              />

              {!isProcessing && (
                <div className="flex justify-center">
                  <button
                    onClick={handleReset}
                    className="px-6 py-3 bg-gray-600 hover:bg-gray-700 text-white font-semibold rounded-lg transition-colors"
                  >
                    Process Another Image
                  </button>
                </div>
              )}
            </div>
          )}
        </div>

        <ImageGallery
          images={gallery}
          onDelete={handleGalleryDelete}
          onDownload={handleGalleryDownload}
        />

        <footer className="text-center mt-12 text-sm text-gray-500">
          <p>Powered by ClipDrop API</p>
        </footer>
      </div>
    </div>
  );
}

export default App;
